import express, { Request, Response } from "express";
import { fromFileUrl, dirname, join } from "@std/path";
import process from "node:process";
import { PrismaClient } from "./prisma/generated/client.ts";

const __filename = fromFileUrl(import.meta.url);
const __dirname = dirname(__filename);

const prisma = new PrismaClient();

const app = express();
app.use(express.json());
const port = process.env.PORT || 3000;
// serve the `public` folder
app.use(express.static(join(__dirname, "public")));

app.get("/", (_req: Request, res: Response) => {
  res.send("Hello, World from Express!");
});

app.get("/students", async (_req: Request, res: Response) => {
  const students = await prisma.student.findMany();
  res.json(students);
});

app.get("/students/:id", async (req: Request, res: Response) => {
  const id = parseInt(req.params.id);
  const student = await prisma.student.findUnique({ where: { id } });

  if (!student) {
    return res.status(404).json({ error: "Student not found" });
  }

  res.json(student);
});

app.post("/students", async (req: Request, res: Response) => {
  const { name, course } = req.body;

  if (!name || !course) {
    return res.status(400).json({ error: "Name and course are required!" });
  }

  const newStudent = await prisma.student.create({ data: { name, course } });
  res.status(201).json(newStudent);
});

app.put("/students/:id", async (req: Request, res: Response) => {
  const id = parseInt(req.params.id);
  const { name, course } = req.body;

  if (!name || !course) {
    return res.status(400).json({ error: "Name and course are required!" });
  }

  const existing = await prisma.student.findUnique({ where: { id } });
  if (!existing) {
    return res.status(404).json({ error: "Student not found" });
  }

  const updatedStudent = await prisma.student.update({ where: { id }, data: { name, course } });
  res.json(updatedStudent);
});

app.patch("/students/:id", async (req: Request, res: Response) => {
  const id = parseInt(req.params.id);
  const { name, course } = req.body;

  const existing = await prisma.student.findUnique({ where: { id } });
  if (!existing) {
    return res.status(404).json({ error: "Student not found" });
  }

  const data: { name?: string; course?: string } = {};
  if (name !== undefined) data.name = name;
  if (course !== undefined) data.course = course;

  const updatedStudent = await prisma.student.update({ where: { id }, data });
  res.json(updatedStudent);
});

app.delete("/students/:id", async (req: Request, res: Response) => {
  const id = parseInt(req.params.id);

  const existing = await prisma.student.findUnique({ where: { id } });
  if (!existing) {
    return res.status(404).json({ error: "Student not found" });
  }

  await prisma.student.delete({ where: { id } });
  return res.status(204).send(); // No Content
});

app.listen(port, () => {
  console.log(`Server running at http://localhost:${port}`);
});